definition module StdReceiver


//	********************************************************************************
//	Clean Standard Object I/O library, version 1.0.1
//	
//	StdReceiver specifies all receiver operations.
//	********************************************************************************


import	StdReceiverDef, StdMaybe
from	iostate	import	PSt, IOSt
from	id		import	RId, R2Id, RIdtoId, R2IdtoId, ==


//	Operations on the ReceiverDevice. 

//	Open uni- and bi-directional receivers:

class Receivers rdef where
	openReceiver	:: .ls !(rdef .ls (PSt .l .p)) !(PSt .l .p)
								   -> (!ErrorReport,!PSt .l .p)
	reopenReceiver	:: .ls !(rdef .ls (PSt .l .p)) !(PSt .l .p)
								   -> (!ErrorReport,!PSt .l .p)
	getReceiverType	::     .(rdef .ls .ps)	-> ReceiverType
/*	openReceiver
		opens the given receiver if no receiver currently exists with the given 
		R(2)Id. The R(2)Id has to be used to send messages to this receiver. 
	reopenReceiver
		first closes the receiver with the same R(2)Id if present, and then opens a 
		new receiver with the given receiver definition. The R(2)Id has to be used 
		to send messages to the new receiver.
	getReceiverType
		returns the type of the receiver (see also getReceivers).
*/

instance Receivers (Receiver  msg)
instance Receivers (Receiver2 msg resp)


closeReceiver		:: !Id !(IOSt .l .p) -> IOSt .l .p
/*	closeReceiver closes the indicated uni- or bi-directional receiver.
	Invalid Ids have no effect.
*/


getReceivers		:: !(IOSt .l .p) -> (![(Id,ReceiverType)], !IOSt .l .p)
/*	getReceivers returns the Ids and ReceiverTypes of all currently open uni- or 
	bi-directional receivers of this interactive process.
*/


enableReceivers			:: ![Id] !(IOSt .l .p) ->                      IOSt .l .p
disableReceivers		:: ![Id] !(IOSt .l .p) ->                      IOSt .l .p
getReceiverSelectState	:: ! Id	 !(IOSt .l .p) -> (!Maybe SelectState,!IOSt .l .p)
/*	(en/dis)ableReceivers
		(en/dis)able the indicated uni- or bi-directional receivers. 
		Note that this implies that in case of synchronous message passing messages 
		can fail (see the comments of syncSend and syncSend2 below). Invalid Ids 
		have no effect. 
	getReceiverSelectState
		yields the current SelectState of the indicated receiver. In case the 
		receiver does not exist, Nothing is returned.
*/


//	Inter-process communication:

//	Message passing status report:
::	SendReport
	=	SendOk
	|	SendUnknownProcess
	|	SendUnknownReceiver
	|	SendUnableReceiver
	|	SendDeadlock

instance toString SendReport

asyncSend :: !(RId msg) msg !(PSt .l .p) -> (!SendReport, !PSt .l .p)
/*	asyncSend posts a message to the receiver indicated by the argument RId. In case
	the indicated receiver belongs to this process, the message is simply buffered. 
	asyncSend is asynchronous: the message will at some point be received by the 
	indicated receiver. 
	The SendReport can be one of the following alternatives:
	-	SendOk:	No exceptional situation has occurred. The message has been sent. 
				Note that even though the message has been sent, it cannot be 
				guaranteed that the message will actually be handled by the 
				indicated receiver because it might become closed, forever disabled,
				or flooded with synchronous messages.
	-	SendUnknownProcess:
				The indicated interactive process does not exist, therefore the 
				receiver also does not exist.
	-	SendUnknownReceiver:
				The indicated receiver does not exist, although the interactive 
				process that created it does exist. 
	-	SendUnableReceiver:
				Does not occur: the message is always buffered, regardless whether 
				the indicated receiver is Able or Unable. Note that in case the 
				receiver never becomes Able, the message will not be handled.
	-	SendDeadlock:
				Does not occur.
*/

syncSend :: !(RId msg) msg !(PSt .l .p) -> (!SendReport, !PSt .l .p)
/*	syncSend posts a message to the receiver indicated by the argument RId. In case 
	the indicated receiver belongs to the current process, the corresponding 
	ReceiverFunction is applied directly to the message argument and current process
	state. 
	syncSend is synchronous: this interactive process blocks evaluation until the 
	indicated receiver has received the message. 
	The SendReport can be one of the following alternatives:
	-	SendOk:	No exceptional situation has occurred. The message has been sent and
				handled by the indicated receiver. 
	-	SendUnknownProcess:
				The indicated interactive process does not exist, therefore the 
				receiver also does not exist.
	-	SendUnknownReceiver:
				The indicated receiver does not exist, although the interactive 
				process that created it does exist. 
	-	SendUnableReceiver:
				The addressee exists, but its ReceiverSelect attribute is Unable. 
				Message passing is halted. The message is not sent. 
	-	SendDeadlock:
				The addressee is involved in a synchronous, cyclic communication 
				with the current process. Blocking the current process would result 
				in a deadlock situation. Message passing is halted to circumvent the
				deadlock. The message is not sent.
*/

syncSend2 :: !(R2Id msg resp) msg  !(PSt .l .p)
	-> (!(!SendReport,!Maybe resp), !PSt .l .p)
/*	syncSend2 posts a message to the receiver indicated by the argument R2Id. In 
	case the indicated receiver belongs to the current process, the corresponding 
	Receiver2Function is applied directly to the message argument and current 
	process state. 
	syncSend2 is synchronous: this interactive process blocks until the indicated 
	receiver has received the message. 
	The SendReport can be one of the following alternatives:
	-	SendOk:	No exceptional situation has occurred. The message has been sent and
				handled by the indicated receiver. The response of the receiver is 
				returned as well as (Just response).
	-	SendUnknownProcess:
				The indicated interactive process does not exist, therefore the 
				receiver also does not exist.
	-	SendUnknownReceiver:
				The indicated receiver does not exist, although the interactive 
				process that created it does exist. 
	-	SendUnableReceiver:
				The addressee exists, but its ReceiverSelect attribute is Unable. 
				Message passing is halted. The message is not sent. 
	-	SendDeadlock:
				The addressee is involved in a synchronous, cyclic communication 
				with the current process. Blocking the current process would result 
				in a deadlock situation. Message passing is halted to circumvent the
				deadlock. The message is not sent.
	In all other cases than SendOk, the optional response is Nothing.
*/
